package b72_j21ref.c28_lambda;
/* M:/72_Java/b72_j21ref/c28_lambda/BlinkingLight.java
 * Příliš žluťoučký kůň úpěl ďábelské ó - PŘÍLIŠ ŽLUŤOUČKÝ KŮŇ ÚPĚL ĎÁBELSKÉ Ó
 */

import ruplib.canvasmanager.Ellipse;
import ruplib.canvasmanager.ICMShape;
import ruplib.canvasmanager.Painter;
import ruplib.geom.Position;
import ruplib.util.IColorable;
import ruplib.util.IO;
import ruplib.util.NamedColor;
import ruplib.util.Repeater;

import java.util.function.Supplier;



/*******************************************************************************
 * Instance třídy {@code BlinkingLight} představují světla,
 * která je možné rozsvítit a zhasnout
 * a která jsou schopna nezávislého blikání,
 * přičemž je možno při konstrukci světla zadat tvar představující žárovku.
 *
 * @author  Rudolf PECINOVSKÝ
 * @version 2023_Summer
 */
public class BlinkingLight implements ICMShape
{
//\CC== CLASS CONSTANTS (CONSTANT CLASS/STATIC ATTRIBUTES/FIELDS) ==============

    /** Implicitní perioda bliknutí světla zadaná v milisekundách
     * (doba, po kterou je světlo rozsvícené a následně zhasnuté). */
    private static final int DEFAULT_PERIOD = 1000;

    /** Implicitní průměr světla. */
    private static final int DEFAULT_SIZE = 50;

    /** Implicitní barva vytvářených instancí. */
    private static final NamedColor DEFAULT_COLOR = NamedColor.YELLOW;

    /** Implicitní továrna na žárovky. */
    private static final Supplier<Ellipse> DEFAULT_FACTORY = Ellipse::new;



//##############################################################################
//\IC== INSTANCE CONSTANTS (CONSTANT INSTANCE ATTRIBUTES/FIELDS) ===============

    /** Barva rozsvíceného světla. */
    private final NamedColor color;

    /** Obrazec představující žárovku schopnou se nechat zobrazit
     *  správcem plátna, zkopírovat se a měnit pozici a rozměr. */
    private final ICMShape cmShapeBulb;

    /** Obrazec představující žárovku schopnou měnit svoji barvu. */
    private final IColorable colorableBulb;

    /** Tovární objekt na výrobu žárovek. */
    private final Supplier<?> bulbFactory;



//\IV== INSTANCE VARIABLES (VARIABLE INSTANCE ATTRIBUTES/FIELDS) ===============

    /** Opakovač sloužící k realizaci zadaného počtu bliknutí
     *  nezdržujícího zbytek programu. */
    private Repeater repeater = null;



//##############################################################################
//\II== INSTANCE INITIALIZERS (CONSTRUCTORS) ===================================

    /***************************************************************************
     * Vytvoří kruhové světlo implicitních rozměrů a barvy
     * umístěné na zadaných souřadnicích.
     *
     * @param x  Vodorovná souřadnice vytvářeného světla
     * @param y  Svislá souřadnice vytvářeného světla
     */
    public BlinkingLight(int x, int y)
    {
        this(x, y, DEFAULT_COLOR);
    }


    /***************************************************************************
     * Vytvoří kruhové světlo implicitních rozměrů
     * zadané barvy rozsvíceného světla umístěné na zadaných souřadnicích.
     *
     * @param x     Vodorovná souřadnice vytvářeného světla
     * @param y     Svislá souřadnice vytvářeného světla
     * @param color Barva rozsvíceného světla
     */
    public BlinkingLight(int x, int y, NamedColor color)
    {
        this(x, y, DEFAULT_SIZE, color);
    }


    /***************************************************************************
     * Vytvoří kruhové světlo zadaných rozměrů a
     * implicitní barvy rozsvíceného světla umístěné na zadaných souřadnicích.
     *
     * @param x    Vodorovná souřadnice vytvářeného světla
     * @param y    Svislá souřadnice vytvářeného světla
     * @param size Průměr vytvářeného světla
     */
    public BlinkingLight(int x, int y, int size)
    {
        this(x, y, size, DEFAULT_COLOR);
    }


    /***************************************************************************
     * Vytvoří kruhové světlo zadaných rozměrů a
     * zadané barvy rozsvíceného světla umístěné na zadaných souřadnicích.
     *
     * @param x     Vodorovná souřadnice vytvářeného světla
     * @param y     Svislá souřadnice vytvářeného světla
     * @param size  Průměr vytvářeného světla
     * @param color Barva rozsvíceného světla
     */
    public BlinkingLight(int x, int y, int size, NamedColor color)
    {
        this(x, y, size, color, DEFAULT_FACTORY);
    }


    /***************************************************************************
     * Vytvoří světlo zadaných rozměrů a
     * zadané barvy rozsvíceného světla umístěné na zadaných souřadnicích,
     * přičemž jeho žárovka je vytvářena pomocí zadané tovární metody,
     * která tak definuje podobu světla.
     *
     * @param <B>         Typ objektu, který bude představovat žárovku
     * @param x           Vodorovná souřadnice vytvářeného světla
     * @param y           Svislá souřadnice vytvářeného světla
     * @param size        Velikost vytvářeného světla
     * @param color       Barva rozsvíceného světla
     * @param bulbFactory Tovární objekt na výrobu žárovek
     */
    public <B extends ICMShape & IColorable>
    BlinkingLight(int x, int y, int size, NamedColor color,
                  Supplier<B> bulbFactory)
    {
        B bulb = bulbFactory.get();

        this.color          = color;
        this.bulbFactory    = bulbFactory;
        this.cmShapeBulb = bulb;
        this.colorableBulb  = bulb;

        cmShapeBulb.setPosition(x, y);
        cmShapeBulb.setSize(size, size);
        colorableBulb .setColor(color);
    }


    /***************************************************************************
     * Tovární metoda vracející kopii daného objektu,
     * tj. instanci s naprosto shodnými vlastnostmi s výjimkou těch,
     * které podle kontraktu shodné být nesmějí - v případě světla vznikne
     * stejně velká, stejně umístěná kopie se stejným tvarem a barvou žárovky.
     * <p>
     * Oproti verzi zděděné z interfejsu {@code IShape} omezuje typ
     * vrácené hodnoty na sebe sama, tj. na {@code BlinkingLight}.
     *
     * @return Požadovaná kopie
     */
    @Override
    @SuppressWarnings("unchecked")
    public BlinkingLight copy()
    {
        Position position = cmShapeBulb.getPosition();
        BlinkingLight copy     = new BlinkingLight(position.x, position.y,
                                      cmShapeBulb.getSize().width, color,
                                      (Supplier)bulbFactory);
        return copy;
    }



//\IA== INSTANCE ABSTRACT METHODS ==============================================
//\IG== INSTANCE GETTERS AND SETTERS ===========================================

    /***************************************************************************
     * Vrátí informaci o tom, je-li světlo rozsvícené.
     *
     * @return  Je-li světlo rozsvícené, vrátí {@code true},
     *          není-li, vrátí {@code false}
     */
    public boolean isOn()
    {
        return color.equals(colorableBulb.getColor());
    }


    /***************************************************************************
     * Vrátí informaci o tom, zda světlo právě bliká.
     *
     * @return Pokud světlo bliká, vrátí {@code true}, jinak vrátí {@code false}
     */
    public boolean isBlinking()
    {
        return repeater != null;
    }


    /***************************************************************************
     * Vrátí barvu rozsvíceného světla.
     *
     * @return  Barva rozsvíceného světla
     */
    public NamedColor getColor()
    {
        return color;
    }


    /***************************************************************************
     * Vrátí x-ovou (vodorovnou) souřadnici pozice instance,
     * tj. vodorovnou souřadnici levého horního rohu opsaného obdélníku.
     *
     * @return  Aktuální vodorovná (x-ová) souřadnice instance,
     *          x=0 má levý okraj plátna, souřadnice roste doprava
     */
//    @Override
    public int getX()
    {
        return cmShapeBulb.getX();
    }


    /***************************************************************************
     * Vrátí y-ovou (svislou) souřadnici pozice instance,
     * tj. svislou souřadnici levého horního rohu opsaného obdélníku.
     *
     * @return  Aktuální svislá (y-ová) souřadnice instance,
     *          y=0 má horní okraj plátna, souřadnice roste dolů
     */
    @Override
    public int getY()
    {
        return cmShapeBulb.getY();
    }


    /***************************************************************************
     * Přemístí instanci na zadanou pozici.
     * Pozice instance je přitom definována jako pozice
     * levého horního rohu opsaného obdélníku.
     *
     * @param x  Nově nastavovaná vodorovná (x-ová) souřadnice instance,
     *           x=0 má levý okraj plátna, souřadnice roste doprava
     * @param y  Nově nastavovaná svislá (y-ová) souřadnice instance,
     *           y=0 má horní okraj plátna, souřadnice roste dolů
     */
    @Override
    public void setPosition(int x, int y)
    {
        cmShapeBulb.setPosition(x, y);
    }


    /***************************************************************************
     * Vrátí aktuální šířku instance.
     * Šířka instance jsou přitom definována jako šířka
     * opsaného obdélníku.
     *
     * @return  Aktuální šířka instance
     */
    @Override
    public int getWidth()
    {
        return cmShapeBulb.getWidth();
    }


    /***************************************************************************
     * Vrátí aktuální výšku instance.
     * Výška instance jsou přitom definována jako výška
     * opsaného obdélníku.
     *
     * @return  Aktuální výška instance
     */
    @Override
    public int getHeight()
    {
        return cmShapeBulb.getHeight();
    }


    /***************************************************************************
     * Nastaví nové rozměry instance.
     * Rozměry instance jsou přitom definovány jako rozměry
     * opsaného obdélníku.
     * Nastavované rozměry musí být nezáporné,
     * místo nulového rozměru se nastaví rozměr rovný jedné.
     *
     * @param width   Nově nastavovaná šířka; šířka &gt;= 0
     * @param height  Nově nastavovaná výška; výška &gt;= 0
     */
   @Override
   public void setSize(int width, int height)
    {
        cmShapeBulb.setSize(width, height);
    }



//\IM== INSTANCE REMAINING NON-PRIVATE METHODS =================================

    /***************************************************************************
     * Světlo se na půl vteřiny rozsvítí a pak zase na půl vteřiny zhasne.
     */
    public void blink()
    {
        switchOn();     System.out.println("ON - " + this);
        IO.pause(DEFAULT_PERIOD / 2);
        switchOff();    System.out.println("OFF - " + this);
        IO.pause(DEFAULT_PERIOD / 2);
    }


    /***************************************************************************
     * Světlo {@code times}-krát zabliká s periodou jedna sekunda;
     * pokaždé se na půl sekundy rozsvítí a na půl sekundy zhasne.
     * Akce probíhá na pozadí, takže nijak neovlivňuje běh okolního programu.
     *
     * @param times Kolikrát světlo blikne
     */
    public void blink(int times)
    {
        blink(times, DEFAULT_PERIOD);
    }


    /***************************************************************************
     * Světlo {@code times}-krát zabliká s periodou {@code period};
     * pokaždé se na půl periody rozsvítí a na druhou půlku zhasne.
     * Akce probíhá na pozadí, takže nijak neovlivňuje běh okolního programu.
     * Metoda neřeší problém s periodou zadanou jako liché číslo,
     * protože smysluplná doba rozsvícení a zhasnutí
     * výrazně převyšuje jednu milisekundu.
     *
     * @param times   Kolikrát světlo blikne
     * @param period  Perioda blikání zadaná v milisekundách
     */
    public void blink(int times, int period)
    {
        blink(times, period, () -> {});
    }


    /***************************************************************************
     * Světlo {@code times}-krát zabliká s periodou {@code period};
     * pokaždé se na půl periody rozsvítí a na druhou půlku zhasne.
     * Akce probíhá na pozadí, takže nijak neovlivňuje běh okolního programu.
     * Metoda neřeší problém s periodou zadanou jako liché číslo,
     * protože smysluplná doba rozsvícení a zhasnutí
     * výrazně převyšuje jednu milisekundu.
     *
     * @param times    Kolikrát světlo blikne
     * @param period   Perioda blikání zadaná v milisekundách
     * @param finished Akce, která se má provést po skončení blikání
     */
    public void blink(int times, int period, Runnable finished)
    {
        assert (! isBlinking()) : //Abych mohl spustit blikání, nesmí blikat
               "Blikání bylo spuštěno před ukončením předchozího";
        repeater = new Repeater();

        Runnable action = () -> {
            switchOn();     IO.pause(period/2);
            switchOff();    IO.pause(period/2);
        };

        Runnable blinkingFinished = () -> {
            repeater = null;
            finished.run();
        };

        repeater.repeat(times, action, blinkingFinished);

    }


    /***************************************************************************
     * Zastaví případné blikání.
     */
    public void stopBlinking()
    {
        assert (repeater != null) :
               "Pokus o ukončení neexistujícího blikání";
        repeater.stop();
        repeater = null;
    }


    /***************************************************************************
     * Prostřednictvím dodaného kreslítka vykreslí obraz své instance.
     *
     * @param painter Kreslítko, které nakreslí instanci
     */
    @Override
    public void paint(Painter painter)
    {
        cmShapeBulb.paint(painter);
    }


    /***************************************************************************
     * Zhasne dané světlo, tj. vybarví je barvou pro zhasnuté světlo.
     */
    public void switchOff()
    {
        colorableBulb.setColor(NamedColor.BLACK);
    }


    /***************************************************************************
     * Rozsvítí dané světlo, tj. vybarví je barvou rozsvíceného světla.
     */
    public void switchOn()
    {
        colorableBulb.setColor(color);
    }


    /***************************************************************************
     * Vrátí textový podpis instance, tj. její řetězcovou reprezentaci.
     *
     * @return Textová reprezentace (podpis) dané instance =
     *         název třídy následovaný závorkami s pozicí, velikostí,
     *         barvou a informací, zda je světlo právě rozsvícené
     */
    @Override
    public String toString()
    {
        return "BlinkingLight{" + getPosition() + ", size=" + getSize() +
               ", color=" + color + ", on=" + isOn() + '}';
    }



//\IP== INSTANCE PRIVATE AND AUXILIARY METHODS =================================



//##############################################################################
//\NT== NESTED DATA TYPES ======================================================
}
